<?php
// Enable error logging (disable display in production)
ini_set('display_errors', 0); // Atur ke 0 untuk produksi
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log'); // Log error ke file ini

// Set headers for API
header('Access-Control-Allow-Origin: *'); // Izinkan semua origin (untuk pengembangan)
header('Access-Control-Allow-Methods: GET, POST, DELETE, PUT, OPTIONS'); // Izinkan metode yang relevan
header('Access-Control-Allow-Headers: Content-Type'); // Izinkan header Content-Type
header('Content-Type: application/json'); // Set Content-Type respons sebagai JSON

// Handle CORS preflight (permintaan OPTIONS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Database connection
$host = 'localhost';
$user = 'rhlqyjgg_admin';
$pass = 'xub!.&%O1pOU';
$db   = 'rhlqyjgg_dbbengkel';

$conn = new mysqli($host, $user, $pass, $db);
$conn->set_charset('utf8mb4'); // Pastikan koneksi menggunakan UTF-8
if ($conn->connect_error) {
    http_response_code(500); // Internal Server Error
    error_log('Database connection failed: ' . $conn->connect_error);
    echo json_encode(['status' => 'error', 'message' => 'Database connection failed']);
    exit;
}

// Configuration untuk URL gambar
// PASTIKAN BASE URL DAN UPLOAD URL INI SAMA PERSIS DENGAN YANG DIGUNAKAN DI FRONTEND DAN API LAINNYA
$baseUrl = 'https://api-bengkel.sumberbarokahsentosa.my.id'; // Tanpa trailing slash
$uploadDir = __DIR__ . '/Uploads/'; // Path fisik di server tempat file diunggah
$uploadUrl = '/Uploads/'; // Path relatif untuk URL publik, dengan leading dan trailing slash

// Fungsi untuk mengirim respons JSON dan menghentikan eksekusi
function respond($status, $message, $extra = []) {
    http_response_code($status === 'success' ? 200 : 400); // Default 200 untuk success, 400 untuk error
    echo json_encode(array_merge(['status' => $status, 'message' => $message], $extra));
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];

// --- GET Method (untuk mengambil data barang) ---
// Note: Bagian ini seharusnya di file `readbarang.php`, tapi jika Anda menggunakannya di sini
// sebagai endpoint multifungsi, ini adalah penanganannya.
if ($method === 'GET') {
    // Parameter filter
    $sparepart_type_id = isset($_GET['sparepart_type_id']) ? intval($_GET['sparepart_type_id']) : 0;
    $tire_type = isset($_GET['tire_type']) ? $conn->real_escape_string($_GET['tire_type']) : '';

    $query = "
        SELECT
            s.id,
            s.nama,
            s.sparepart_type_id,
            st.nama AS sparepart_type_name,
            s.ukuran,
            s.merk,
            s.jenis_kendaraan,
            s.harga_grosir,
            s.harga_eceran,
            s.stok,
            s.id_suplier,
            su.nama_suplier AS suplier_name,
            s.foto,
            s.tire_type
        FROM sparepart s
        LEFT JOIN sparepart_types st ON s.sparepart_type_id = st.id
        LEFT JOIN suplier su ON s.id_suplier = su.id
    ";

    $conditions = [];
    $params = [];
    $types = '';

    if ($sparepart_type_id > 0) {
        $conditions[] = 's.sparepart_type_id = ?';
        $params[] = $sparepart_type_id;
        $types .= 'i';
    }

    if (!empty($tire_type)) {
        $conditions[] = 's.tire_type = ?';
        $params[] = $tire_type;
        $types .= 's';
    }

    if (!empty($conditions)) {
        $query .= ' WHERE ' . implode(' AND ', $conditions);
    }

    $query .= ' ORDER BY s.id DESC';

    $stmt = $conn->prepare($query);
    if (!$stmt) {
        error_log('Prepare error for GET: ' . $conn->error);
        respond('error', 'Database query preparation failed for GET');
    }

    if (!empty($params)) {
        // Dynamic binding for bind_param
        $bind_names = array($types);
        for ($i = 0; $i < count($params); $i++) {
            $bind_name = 'bind' . $i;
            $$bind_name = $params[$i];
            $bind_names[] = &$$bind_name;
        }
        call_user_func_array(array($stmt, 'bind_param'), $bind_names);
    }

    $stmt->execute();
    $result = $stmt->get_result();

    if (!$result) {
        error_log('Execution error for GET: ' . $stmt->error);
        respond('error', 'Database query execution failed for GET');
    }

    $data = [];
    while ($row = $result->fetch_assoc()) {
        // Type casting
        $row['id'] = (int)$row['id'];
        $row['sparepart_type_id'] = (int)$row['sparepart_type_id'];
        $row['harga_grosir'] = (float)$row['harga_grosir'];
        $row['harga_eceran'] = (float)$row['harga_eceran'];
        $row['stok'] = (int)$row['stok'];
        $row['id_suplier'] = isset($row['id_suplier']) ? (int)$row['id_suplier'] : null;
        
        // Pastikan foto adalah URL lengkap untuk respons GET
        // Jika sudah URL lengkap, biarkan. Jika tidak, tambahkan base URL dan upload URL.
        if (!empty($row['foto']) && substr($row['foto'], 0, 4) !== 'http') {
            // Ambil hanya nama file jika yang tersimpan di DB adalah path fisik
            $filename_from_db = basename($row['foto']);
            $row['foto'] = $baseUrl . $uploadUrl . $filename_from_db;
        } elseif (empty($row['foto'])) {
             $row['foto'] = null; // Pastikan null jika kosong
        }
        
        $data[] = $row;
    }
    $stmt->close();
    respond('success', 'Data retrieved successfully', ['data' => $data]);
}

// --- POST Method (untuk menambahkan data barang baru) ---
elseif ($method === 'POST') {
    // Ambil dan sanitasi input
    $nama = $conn->real_escape_string($_POST['nama'] ?? '');
    $sparepart_type_id = intval($_POST['sparepart_type_id'] ?? 0);
    $ukuran = $conn->real_escape_string($_POST['ukuran'] ?? '');
    $merk = $conn->real_escape_string($_POST['merk'] ?? '');
    $jenis_kendaraan = $conn->real_escape_string($_POST['jenis_kendaraan'] ?? '');
    $harga_grosir = floatval($_POST['harga_grosir'] ?? 0);
    $harga_eceran = floatval($_POST['harga_eceran'] ?? 0);
    $stok = intval($_POST['stok'] ?? 0);
    $id_suplier = !empty($_POST['id_suplier']) ? intval($_POST['id_suplier']) : null;
    $tire_type = $conn->real_escape_string($_POST['tire_type'] ?? '');

    // Validasi input
    if (empty($nama) || !preg_match('/^[a-zA-Z0-9\s]+$/', $nama)) {
        respond('error', 'Nama hanya boleh berisi huruf, angka, dan spasi');
    }
    if ($sparepart_type_id <= 0) {
        respond('error', 'ID tipe sparepart tidak valid');
    }
    if ($harga_grosir < 0 || $harga_eceran < 0) {
        respond('error', 'Harga tidak boleh negatif');
    }
    if ($stok < 0) {
        respond('error', 'Stok tidak boleh negatif');
    }

    // Validasi keberadaan sparepart_type_id
    $stmt = $conn->prepare('SELECT id FROM sparepart_types WHERE id = ?');
    if (!$stmt) {
        error_log('Prepare error (sparepart_types check): ' . $conn->error);
        respond('error', 'Database error preparing sparepart type check');
    }
    $stmt->bind_param('i', $sparepart_type_id);
    $stmt->execute();
    if ($stmt->get_result()->num_rows === 0) {
        respond('error', 'ID tipe sparepart tidak ditemukan');
    }
    $stmt->close();

    // Validasi keberadaan id_suplier (jika ada)
    if ($id_suplier !== null) {
        $stmt = $conn->prepare('SELECT id FROM suplier WHERE id = ?');
        if (!$stmt) {
            error_log('Prepare error (suplier check): ' . $conn->error);
            respond('error', 'Database error preparing suplier check');
        }
        $stmt->bind_param('i', $id_suplier);
        $stmt->execute();
        if ($stmt->get_result()->num_rows === 0) {
            respond('error', 'ID suplier tidak ditemukan');
        }
        $stmt->close();
    }

    $foto_url_to_db = null; // Variabel ini akan menyimpan URL yang akan disimpan ke DB
    if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
        // Pastikan direktori unggahan ada dan dapat ditulis
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                respond('error', 'Gagal membuat direktori unggahan');
            }
        }
        if (!is_writable($uploadDir)) {
            respond('error', 'Direktori unggahan tidak dapat ditulis');
        }

        // Validasi tipe dan ukuran file
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
        $maxFileSize = 2 * 1024 * 1024; // 2 MB
        $fileType = mime_content_type($_FILES['foto']['tmp_name']);
        $fileSize = $_FILES['foto']['size'];

        if (!in_array($fileType, $allowedTypes)) {
            respond('error', 'Hanya file JPEG, PNG, atau GIF yang diizinkan');
        }
        if ($fileSize > $maxFileSize) {
            respond('error', 'Ukuran file melebihi batas 2MB');
        }

        // Bentuk nama file unik dan path target fisik
        $fileExtension = pathinfo($_FILES['foto']['name'], PATHINFO_EXTENSION);
        $fileName = 'sparepart_' . time() . '_' . uniqid() . '.' . $fileExtension; // Tambahkan uniqid untuk lebih unik
        $targetPath = $uploadDir . $fileName;

        // Pindahkan file yang diunggah ke direktori target fisik
        if (!move_uploaded_file($_FILES['foto']['tmp_name'], $targetPath)) {
            error_log('Gagal mengunggah file: ' . $_FILES['foto']['name'] . ' ke ' . $targetPath);
            respond('error', 'Gagal mengunggah file ke server');
        }
        
        // INI ADALAH PERUBAHAN KRITIS: SIMPAN URL LENGKAP KE DATABASE
        $foto_url_to_db = $baseUrl . $uploadUrl . $fileName; 
    }

    // Siapkan query INSERT
    $stmt = $conn->prepare('INSERT INTO sparepart (nama, sparepart_type_id, ukuran, merk, jenis_kendaraan, harga_grosir, harga_eceran, stok, id_suplier, foto, tire_type) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)');
    if (!$stmt) {
        error_log('Prepare error (INSERT): ' . $conn->error);
        respond('error', 'Database error preparing insert query');
    }

    // Bind parameter, gunakan $foto_url_to_db yang berisi URL lengkap
    $stmt->bind_param('sisssddiiss', 
        $nama, 
        $sparepart_type_id, 
        $ukuran, 
        $merk, 
        $jenis_kendaraan, 
        $harga_grosir, 
        $harga_eceran, 
        $stok, 
        $id_suplier, 
        $foto_url_to_db, // Ini yang sekarang menyimpan URL lengkap
        $tire_type
    );

    // Eksekusi statement
    if ($stmt->execute()) {
        $new_id = $conn->insert_id;
        respond('success', 'Sparepart berhasil ditambahkan', ['id' => $new_id, 'foto' => $foto_url_to_db]); // Sertakan URL foto dalam respons
    } else {
        error_log('Execute error (INSERT): ' . $stmt->error);
        respond('error', 'Gagal menambahkan sparepart: ' . $stmt->error);
    }
    $stmt->close();
}

// --- PUT Method (Update Sparepart) ---
elseif ($method === 'PUT') {
    $input = json_decode(file_get_contents('php://input'), true);

    $id = intval($input['id'] ?? 0);
    if ($id <= 0) {
        respond('error', 'ID sparepart tidak valid');
    }

    $set_clauses = [];
    $params = [];
    $types = '';

    // Ambil data yang sudah ada untuk foto (jika tidak diupdate)
    $current_foto = null;
    $stmt_fetch = $conn->prepare("SELECT foto FROM sparepart WHERE id = ?");
    if (!$stmt_fetch) {
        error_log('Prepare error (fetch current foto): ' . $conn->error);
        respond('error', 'Error preparing to fetch current photo');
    }
    $stmt_fetch->bind_param('i', $id);
    $stmt_fetch->execute();
    $result_fetch = $stmt_fetch->get_result();
    if ($result_fetch->num_rows > 0) {
        $current_foto = $result_fetch->fetch_assoc()['foto'];
    }
    $stmt_fetch->close();

    // Handle file upload for update
    $foto_url_to_db = $current_foto; // Default to current photo URL
    if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
        if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);
        if (!is_writable($uploadDir)) respond('error', 'Direktori unggahan tidak dapat ditulis');

        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
        $maxFileSize = 2 * 1024 * 1024;
        $fileType = mime_content_type($_FILES['foto']['tmp_name']);
        $fileSize = $_FILES['foto']['size'];

        if (!in_array($fileType, $allowedTypes)) respond('error', 'Hanya file JPEG, PNG, atau GIF yang diizinkan');
        if ($fileSize > $maxFileSize) respond('error', 'Ukuran file melebihi batas 2MB');

        // Hapus foto lama jika ada dan berbeda dari default placeholder
        if ($current_foto && substr($current_foto, 0, 4) === 'http' && strpos($current_foto, $baseUrl . $uploadUrl) !== false) {
            $old_filename = basename($current_foto);
            $old_filepath = $uploadDir . $old_filename;
            if (file_exists($old_filepath) && is_file($old_filepath)) {
                if (!unlink($old_filepath)) {
                    error_log('Failed to delete old photo file: ' . $old_filepath);
                    // Lanjutkan eksekusi meskipun gagal hapus file lama (opsional: bisa respond error)
                }
            }
        }

        $fileExtension = pathinfo($_FILES['foto']['name'], PATHINFO_EXTENSION);
        $fileName = 'sparepart_' . time() . '_' . uniqid() . '.' . $fileExtension;
        $targetPath = $uploadDir . $fileName;
        
        if (!move_uploaded_file($_FILES['foto']['tmp_name'], $targetPath)) {
            error_log('Gagal mengunggah file baru: ' . $_FILES['foto']['name']);
            respond('error', 'Gagal mengunggah file foto baru');
        }
        $foto_url_to_db = $baseUrl . $uploadUrl . $fileName; // Simpan URL lengkap
    }

    // Bangun query update berdasarkan input yang diterima
    if (isset($input['nama'])) {
        $set_clauses[] = 'nama = ?';
        $params[] = $conn->real_escape_string($input['nama']);
        $types .= 's';
    }
    if (isset($input['sparepart_type_id'])) {
        $set_clauses[] = 'sparepart_type_id = ?';
        $params[] = intval($input['sparepart_type_id']);
        $types .= 'i';
    }
    if (isset($input['ukuran'])) {
        $set_clauses[] = 'ukuran = ?';
        $params[] = $conn->real_escape_string($input['ukuran']);
        $types .= 's';
    }
    if (isset($input['merk'])) {
        $set_clauses[] = 'merk = ?';
        $params[] = $conn->real_escape_string($input['merk']);
        $types .= 's';
    }
    if (isset($input['jenis_kendaraan'])) {
        $set_clauses[] = 'jenis_kendaraan = ?';
        $params[] = $conn->real_escape_string($input['jenis_kendaraan']);
        $types .= 's';
    }
    if (isset($input['harga_grosir'])) {
        $set_clauses[] = 'harga_grosir = ?';
        $params[] = floatval($input['harga_grosir']);
        $types .= 'd';
    }
    if (isset($input['harga_eceran'])) {
        $set_clauses[] = 'harga_eceran = ?';
        $params[] = floatval($input['harga_eceran']);
        $types .= 'd';
    }
    if (isset($input['stok'])) {
        $set_clauses[] = 'stok = ?';
        $params[] = intval($input['stok']);
        $types .= 'i';
    }
    if (isset($input['id_suplier'])) {
        $set_clauses[] = 'id_suplier = ?';
        $params[] = !empty($input['id_suplier']) ? intval($input['id_suplier']) : null;
        $types .= 'i';
    }
    if (isset($input['tire_type'])) {
        $set_clauses[] = 'tire_type = ?';
        $params[] = $conn->real_escape_string($input['tire_type']);
        $types .= 's';
    }
    // Tambahkan foto ke klausa SET jika ada perubahan foto
    if (isset($_FILES['foto']) && $_FILES['foto']['error'] === UPLOAD_ERR_OK) {
        $set_clauses[] = 'foto = ?';
        $params[] = $foto_url_to_db;
        $types .= 's';
    } elseif (isset($input['foto']) && $input['foto'] === null) {
        // Jika frontend secara eksplisit mengirim foto: null, set foto di DB ke null
        $set_clauses[] = 'foto = ?';
        $params[] = null;
        $types .= 's';
    }

    if (empty($set_clauses)) {
        respond('success', 'Tidak ada data untuk diperbarui');
    }

    $query_update = 'UPDATE sparepart SET ' . implode(', ', $set_clauses) . ' WHERE id = ?';
    $params[] = $id; // ID adalah parameter terakhir
    $types .= 'i'; // Tipe untuk ID

    $stmt = $conn->prepare($query_update);
    if (!$stmt) {
        error_log('Prepare error (UPDATE): ' . $conn->error);
        respond('error', 'Database error preparing update query');
    }

    // Dynamic binding for bind_param
    $bind_names = array($types);
    for ($i = 0; $i < count($params); $i++) {
        $bind_name = 'bind' . $i;
        $$bind_name = $params[$i];
        $bind_names[] = &$$bind_name;
    }
    call_user_func_array(array($stmt, 'bind_param'), $bind_names);

    if ($stmt->execute()) {
        if ($stmt->affected_rows > 0) {
            respond('success', 'Sparepart berhasil diperbarui');
        } else {
            respond('success', 'Tidak ada perubahan yang dilakukan atau sparepart tidak ditemukan');
        }
    } else {
        error_log('Execute error (UPDATE): ' . $stmt->error);
        respond('error', 'Gagal memperbarui sparepart: ' . $stmt->error);
    }
    $stmt->close();
}

// --- DELETE Method (Hapus Sparepart) ---
elseif ($method === 'DELETE') {
    parse_str($_SERVER['QUERY_STRING'], $params_delete);
    $id = intval($params_delete['id'] ?? 0);

    if ($id <= 0) {
        respond('error', 'ID sparepart tidak valid');
    }

    // Ambil URL foto dari database sebelum menghapus record
    $stmt_foto = $conn->prepare('SELECT foto FROM sparepart WHERE id = ?');
    if (!$stmt_foto) {
        error_log('Prepare error (DELETE - fetch foto): ' . $conn->error);
        respond('error', 'Database error preparing to fetch photo for deletion');
    }
    $stmt_foto->bind_param('i', $id);
    $stmt_foto->execute();
    $result_foto = $stmt_foto->get_result();
    $foto_url_db = null;
    if ($result_foto->num_rows > 0) {
        $foto_url_db = $result_foto->fetch_assoc()['foto'];
    }
    $stmt_foto->close();

    // Hapus record dari database
    $stmt = $conn->prepare('DELETE FROM sparepart WHERE id = ?');
    if (!$stmt) {
        error_log('Prepare error (DELETE): ' . $conn->error);
        respond('error', 'Database error preparing delete query');
    }
    $stmt->bind_param('i', $id);
    if ($stmt->execute()) {
        if ($stmt->affected_rows > 0) {
            // Jika penghapusan DB berhasil, coba hapus file fisik
            if ($foto_url_db && substr($foto_url_db, 0, 4) === 'http' && strpos($foto_url_db, $baseUrl . $uploadUrl) !== false) {
                $filename_from_url = basename($foto_url_db);
                $physical_filepath = $uploadDir . $filename_from_url;
                if (file_exists($physical_filepath) && is_file($physical_filepath)) {
                    if (!unlink($physical_filepath)) {
                        error_log('Failed to delete physical file: ' . $physical_filepath);
                        // Anda bisa memilih untuk tetap merespons sukses ke klien,
                        // atau memberi tahu bahwa file fisik tidak terhapus.
                    }
                }
            }
            respond('success', 'Sparepart berhasil dihapus');
        } else {
            respond('error', 'Sparepart tidak ditemukan atau tidak ada perubahan yang dilakukan');
        }
    } else {
        error_log('Execute error (DELETE): ' . $stmt->error);
        respond('error', 'Gagal menghapus sparepart: ' . $stmt->error);
    }
    $stmt->close();
}

$conn->close();
?>